<?php
declare( strict_types=1 );

namespace WP_Defender;

use WP_Defender\Traits\Defender_Bootstrap;

/**
 * Class Bootstrap
 * @package WP_Defender
 */
class Bootstrap {
	use Defender_Bootstrap;

	/**
	 * Activation.
	 */
	public function activation_hook(): void {
		$this->activation_hook_common();
		$this->set_free_installation_timestamp();
	}

	/**
	 * Add option with plugin install date.
	 *
	 * @since 2.4
	 */
	protected function set_free_installation_timestamp(): void {
		// It's for both cases because don’t have a Pro checking during plugin activation.
		if ( empty( get_site_option( 'defender_free_install_date' ) ) ) {
			update_site_option( 'defender_free_install_date', time() );
		}
	}

	/**
	 * Load all modules.
	 */
	public function init_modules(): void {
		$this->init_modules_common();
		$this->init_free_dashboard();
		$this->init_black_friday();
	}

	/**
	 * @return void
	 */
	public function init_free_dashboard(): void {
		$file_path = defender_path( 'extra/free-dashboard/module.php' );
		if ( file_exists( $file_path ) ) {
			require_once $file_path;

			add_filter( 'wdev-email-message-' . DEFENDER_PLUGIN_BASENAME, [ &$this, 'defender_ads_message' ] );

			$screen_prefix = 'defender';
			$screen_suffix = is_multisite() ? '-network' : '';

			$free_install_date = get_site_option( 'defender_free_install_date', false );

			do_action(
				'wpmudev_register_notices',
				'defender',
				[
					'basename' => DEFENDER_PLUGIN_BASENAME,
					'title' => 'Defender',
					'wp_slug' => 'defender-security',
					'cta_email' => __( 'Get Secure!', 'wpdef' ),
					'mc_list_id' => '0cecf2890e',
					'installed_on' => $free_install_date ?: time(),
					'screens' => [
						'toplevel_page_wp-defender' . $screen_suffix,
						$screen_prefix . '_page_wdf-hardener' . $screen_suffix,
						$screen_prefix . '_page_wdf-scan' . $screen_suffix,
						$screen_prefix . '_page_wdf-logging' . $screen_suffix,
						$screen_prefix . '_page_wdf-ip-lockout' . $screen_suffix,
						$screen_prefix . '_page_wdf-waf' . $screen_suffix,
						$screen_prefix . '_page_wdf-2fa' . $screen_suffix,
						$screen_prefix . '_page_wdf-advanced-tools' . $screen_suffix,
						$screen_prefix . '_page_wdf-notification' . $screen_suffix,
						$screen_prefix . '_page_wdf-setting' . $screen_suffix,
						$screen_prefix . '_page_wdf-tutorial' . $screen_suffix,
					],
				]
			);
		}
	}

	/**
	 * Load the black friday module.
	 *
	 * @return void
	 * @since 3.4.1
	 */
	public function init_black_friday(): void {
		$file_path = defender_path( 'extra/black-friday/banner.php' );
		if ( file_exists( $file_path ) ) {
			require_once $file_path;

			if ( class_exists( 'WPMUDEV\BlackFriday\Banner' ) ) {
				new \WPMUDEV\BlackFriday\Banner(
					[
						'close' => esc_html__( 'Close', 'wpdef' ),
						'get_deal' => esc_html__( 'Get deal', 'wpdef' ),
						'intro' => esc_html__( 'Black Friday offer for WP businesses and agencies', 'wpdef' ),
						'off' => esc_html__( 'Off', 'wpdef' ),
						'title' => esc_html__( 'Everything you need to run your WP business for', 'wpdef' ),
						'discount' => __( '83.5', 'wpdef' ),
						'price' => __( '3000', 'wpdef' ),
						'description' => esc_html__( 'From the creators of Defender, WPMU DEV’s all-in-one platform gives you all the Pro tools and support you need to run and grow a web development business. Trusted by over 50,000 web developers. Limited deals available.', 'wpdef' ),
					],
					'https://wpmudev.com/black-friday/?coupon=BFP-2022&utm_source=defender&utm_medium=plugin&utm_campaign=BFP-2022-defender&utm_id=BFP-2022&utm_term=BF-2022-plugin-Defender&utm_content=BF-2022',
					\WPMUDEV\BlackFriday\Banner::DEFENDER
				);
			}
		}
	}

	/**
	 * @param $message
	 *
	 * @return string
	 */
	public function defender_ads_message( string $message ): string {
		return __( "You're awesome for installing Defender! Are you interested in how to make the most of this plugin? We've collected all the best security resources we know in a single email - just for users of Defender!", 'wpdef' );
	}
}
